<?php 

// === Config === 

define('DBDSE_STORE_URL', 'https://divibooster.com'); // URL of site with EDD store
define('DBDSE_ITEM_ID', 197662); // id of the plugin in EDD
define('DBDSE_SETTINGS_PAGE', 'dbdse_settings');

// === Add settings page for entering the license ===

add_action('admin_menu', 'dbdse_license_menu');

function dbdse_license_menu() {
	add_options_page('Divi Shortcode Enabler', 'Divi Shortcode Enabler', 'manage_options', DBDSE_SETTINGS_PAGE, 'dbdse_license_page');
}

function dbdse_license_page() {
	$license = get_option( 'dbdse_license_key' );
	$status  = get_option( 'dbdse_license_status' );
	?>
	<div class="wrap">
		<h2><?php _e('Divi Shortcode Enabler'); ?></h2>
		<form method="post" action="options.php">

			<?php settings_fields('dbdse_license'); ?>

			<table class="form-table">
				<tbody>
					<tr valign="top">
						<th scope="row" valign="top">
							<?php _e('License Key'); ?>
						</th>
						<td>
							<input id="dbdse_license_key" name="dbdse_license_key" type="password" class="regular-text" value="<?php esc_attr_e( $license ); ?>" />
							<label class="description" for="dbdse_license_key"><?php _e('Enter your license key'); ?></label>
							<p>Get your license key from your <a href="https://divibooster.com/your-account/" target="_blank">Divi Booster account</a>, or <a href="https://divibooster.com/divi-shortcode-enabler/" target="_blank">purchase a license</a>.</p>
						</td>
					</tr>
					<?php if( false !== $license ) { ?>
						<tr valign="top">
							<th scope="row" valign="top">
								<?php _e('Activate License'); ?>
							</th>
							<td>
								<?php wp_nonce_field( 'dbdse_nonce', 'dbdse_nonce' ); ?>
								<?php if( $status !== false && $status == 'valid' ) { ?>
									<span style="color:green;line-height:28px;margin-right:1em;"><?php _e('Active'); ?></span>
									<input type="submit" class="button-secondary" name="edd_license_deactivate" value="<?php _e('Deactivate License'); ?>"/>
								<?php } else { ?>
									<input type="submit" class="button-secondary" name="edd_license_activate" value="<?php _e('Activate License'); ?>"/>
								<?php } ?>
							</td>
						</tr>
					<?php } ?>
				</tbody>
			</table>
			<?php submit_button(); ?>

		</form>
	<?php
}

// === Register the license key option ===

add_action('admin_init', 'dbdse_register_option');

function dbdse_register_option() {
	// creates our settings in the options table
	register_setting('dbdse_license', 'dbdse_license_key', 'dbdse_sanitize_license' );
}

function dbdse_sanitize_license( $new ) {
	$old = get_option( 'dbdse_license_key' );
	if( $old && $old != $new ) {
		delete_option( 'dbdse_license_status' ); // new license has been entered, so must reactivate
	}
	return $new;
}


// === License activation ===

add_action('admin_init', 'dbdse_activate_license');

function dbdse_activate_license() {
	
	// listen for our activate button to be clicked
	if( isset( $_POST['edd_license_activate'] ) ) {
		// run a quick security check
	 	if( ! check_admin_referer( 'dbdse_nonce', 'dbdse_nonce' ) )
			return; // get out if we didn't click the Activate button
		// retrieve the license from the database
		
		$license = trim( get_option( 'dbdse_license_key' ) );

		$response = dbdse_submit_license('activate', $license);
		
		// make sure the response came back okay
		if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
			$message =  ( is_wp_error( $response ) && ! empty( $response->get_error_message() ) ) ? $response->get_error_message() : __( 'An error occurred, please try again.' );
		} else {
			$license_data = json_decode( wp_remote_retrieve_body( $response ) );
			if ( false === $license_data->success ) {
				switch( $license_data->error ) {
					case 'expired' :
						$message = sprintf(
							__( 'Your license key expired on %s.' ),
							date_i18n( get_option( 'date_format' ), strtotime( $license_data->expires, current_time( 'timestamp' ) ) )
						);
						break;
					case 'revoked' :
						$message = __( 'Your license key has been disabled.' );
						break;
					case 'missing' :
						$message = __( 'Invalid license.' );
						break;
					case 'invalid' :
					case 'site_inactive' :
						$message = __( 'Your license is not active for this URL.' );
						break;
					case 'item_name_mismatch' :
						$message = sprintf( __( 'This appears to be an invalid license key for %s.' ), dbdse_ITEM_NAME );
						break;
					case 'no_activations_left':
						$message = __( 'Your license key has reached its activation limit.' );
						break;
					default :
						$message = __( 'An error occurred, please try again.' );
						break;
				}
			}
		}
		// Check if anything passed on a message constituting a failure
		if ( ! empty( $message ) ) {
			$base_url = admin_url('options-general.php?page='.DBDSE_SETTINGS_PAGE);
			$redirect = add_query_arg( array( 'sl_activation' => 'false', 'message' => urlencode( $message ) ), $base_url );
			wp_redirect( $redirect );
			exit();
		}
		// $license_data->license will be either "valid" or "invalid"
		update_option( 'dbdse_license_status', $license_data->license );
		wp_redirect( admin_url( 'options-general.php?page=' . DBDSE_SETTINGS_PAGE) );
		exit();
	}
}


// === License deactivation ===

add_action('admin_init', 'dbdse_deactivate_license');

function dbdse_deactivate_license() {
	
	// listen for our activate button to be clicked
	if( isset( $_POST['edd_license_deactivate'] ) ) {
		// run a quick security check
	 	if( ! check_admin_referer( 'dbdse_nonce', 'dbdse_nonce' ) )
			return; // get out if we didn't click the Activate button
		// retrieve the license from the database
		$license = trim( get_option( 'dbdse_license_key' ) );

		if (!dbdse_license_deactivated_in_edd($license)) {
			
			$response = dbdse_submit_license('deactivate', $license);
						
			// make sure the response came back okay
			if ( is_wp_error( $response ) || 200 !== wp_remote_retrieve_response_code( $response ) ) {
				$message =  ( is_wp_error( $response ) && ! empty( $response->get_error_message() ) ) ? $response->get_error_message() : __( 'An error occurred while connecting to the license server, please try again.' );
				
			} else {
				
				$license_data = json_decode( wp_remote_retrieve_body( $response ) );
				if ( false === $license_data->success ) {
					$message = __( 'An error occurred while deactivating the license, please try again.' );
				}
			}
			// Check if anything passed on a message constituting a failure
			if ( ! empty( $message ) ) {
				$base_url = admin_url('options-general.php?page='.DBDSE_SETTINGS_PAGE);
				$redirect = add_query_arg( array( 'sl_deactivation' => 'false', 'message' => urlencode( $message ) ), $base_url );
				wp_redirect( $redirect );
				exit();
			}
		}
		
		update_option('dbdse_license_status', 'deactivated');
		wp_redirect( admin_url( 'options-general.php?page=' . DBDSE_SETTINGS_PAGE) );
		exit();
	}
}

if (!function_exists('dbdse_license_deactivated_in_edd')) {
	function dbdse_license_deactivated_in_edd($license) {
		$deactivated = false;
		$response = dbdse_submit_license('check', $license);
		if (!is_wp_error($response)) {
			$license_data = json_decode(wp_remote_retrieve_body($response));
			if ($license_data->license === 'inactive') {
				$deactivated = true;			
			}
		} 
		return apply_filters('dbdse_license_deactivated_in_edd', $deactivated, $license);
	}
}

if (!function_exists('dbdse_submit_license')) {
	function dbdse_submit_license($action, $license) {
		$result = wp_remote_post(
			DBDSE_STORE_URL, 
			array(
				'timeout' => 15, 
				'body' => array(
					'edd_action' => $action.'_license',
					'license'    => $license,
					'item_id'    => DBDSE_ITEM_ID,
					'url'        => home_url()
				)
			)
		);
		return apply_filters('dbdse_submit_license', $result, $action, $license);
	}
}

// === Display license activation / deactivation errors to user ===

add_action('admin_notices', 'dbdse_admin_notices');

function dbdse_admin_notices() {
	if ( isset( $_GET['sl_activation'] ) && ! empty( $_GET['message'] ) ) {
		switch( $_GET['sl_activation'] ) {
			case 'false':
				$message = urldecode( $_GET['message'] );
				?>
				<div class="error">
					<p><?php esc_html_e($message); ?></p>
				</div>
				<?php
				break;
			case 'true':
			default:
				// Can put a custom success message here for when activation is successful if they want.
				break;
		}
	}
	if ( isset( $_GET['sl_deactivation'] ) && ! empty( $_GET['message'] ) ) {
		switch( $_GET['sl_deactivation'] ) {
			case 'false':
				$message = urldecode( $_GET['message'] );
				?>
				<div class="error">
					<p>Error: <?php esc_html_e($message); ?></p>
				</div>
				<?php
				break;
			case 'true':
			default:
				// Can put a custom success message here for when activation is successful if they want.
				break;
		}
	}
}

// === Notify admin via plugins.php that license required to activate plugin ===

add_action( 'after_plugin_row_' . plugin_basename(DBDSE_FILE), 'dbdse_after_plugin_row', 20, 3 );

function dbdse_after_plugin_row($plugin_file, $plugin_data, $status) {
	
	// Don't display message if the plugin is correctly licensed
	if (get_option('dbdse_license_status') === 'valid') { 
		return;
	}
	
	// Display license message
	$settings_url = admin_url('options-general.php?page='.DBDSE_SETTINGS_PAGE);
	?>
	<tr class="plugin-update-tr active"><td colspan="3" class="plugin-update colspanchange"><div class="update-message notice inline notice-warning notice-alt"><p>Enter your license key on the <a href="<?php esc_attr_e($settings_url); ?>">settings page</a> to enable updates. You can get your license from your <a href="https://divibooster.com/your-account/purchase-history/" target="_blank">Divi Booster account</a>, or <a href="https://divibooster.com/divi-shortcode-enabler/" target="_blank">purchase a license</a>.</p></div></td>
	</tr>
	<?php
}

add_action('admin_head', 'dbdse_plugins_php_license_required_css');

function dbdse_plugins_php_license_required_css() {
	
	// Make sure we're on plugins.php
	global $pagenow;
	if ($pagenow !== 'plugins.php') { return; }
	
	// Don't add CSS if the plugin is correctly licensed
	if (get_option('dbdse_license_status') === 'valid') { 
		return;
	}
	
	// Remove border bottom after shortcode enabler plugin entry
	?>
	<style>
	.plugins tr[data-slug="divi-shortcode-enabler"].active td, 
	.plugins tr[data-slug="divi-shortcode-enabler"].active th {
		box-shadow: none !important;
	}
	</style>
	<?php
}